<?php
declare(strict_types=1);

function gmap_load_config(): array
{
    static $config = null;
    if ($config !== null) {
        return $config;
    }

    $candidates = [];
    $envPath = getenv('GMAP_CONFIG_PATH');
    if (is_string($envPath) && $envPath !== '') {
        $candidates[] = $envPath;
    }

    $apiDir = __DIR__;
    $candidates[] = $apiDir . '/../_secure/config.php';
    $candidates[] = dirname($apiDir, 2) . '/_secure/config.php';
    $candidates[] = dirname($apiDir, 3) . '/_secure/config.php';

    $documentRoot = (string)($_SERVER['DOCUMENT_ROOT'] ?? '');
    if ($documentRoot !== '') {
        $candidates[] = rtrim($documentRoot, '/\\') . '/_secure/config.php';
        $candidates[] = dirname(rtrim($documentRoot, '/\\')) . '/_secure/config.php';
    }

    $homeDirs = [
        (string)($_SERVER['HOME'] ?? ''),
        (string)getenv('HOME'),
    ];
    foreach ($homeDirs as $home) {
        $home = trim($home);
        if ($home !== '') {
            $candidates[] = rtrim($home, '/\\') . '/_secure/config.php';
        }
    }

    foreach ($candidates as $path) {
        if (!is_string($path) || $path === '') {
            continue;
        }
        $resolved = realpath($path);
        $candidate = ($resolved !== false) ? $resolved : $path;
        if (is_file($candidate) && is_readable($candidate)) {
            $loaded = require $candidate;
            if (is_array($loaded)) {
                $config = $loaded;
                return $config;
            }
        }
    }

    throw new RuntimeException('Secure config file not found. Set GMAP_CONFIG_PATH or place config.php in _secure.');
}

function gmap_pdo(): PDO
{
    static $pdo = null;
    if ($pdo instanceof PDO) {
        return $pdo;
    }

    $cfg = gmap_load_config();
    $charset = $cfg['db_charset'] ?? 'utf8mb4';
    $dsn = sprintf(
        'mysql:host=%s;dbname=%s;charset=%s',
        $cfg['db_host'] ?? 'localhost',
        $cfg['db_name'] ?? '',
        $charset
    );

    $pdo = new PDO(
        $dsn,
        $cfg['db_user'] ?? '',
        $cfg['db_pass'] ?? '',
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ]
    );
    return $pdo;
}

function gmap_hmac_secret(): string
{
    $cfg = gmap_load_config();
    $secret = (string)($cfg['hmac_secret'] ?? '');
    if (strlen($secret) < 32) {
        throw new RuntimeException('hmac_secret must be at least 32 characters.');
    }
    return $secret;
}

function gmap_admin_username(): string
{
    $cfg = gmap_load_config();
    return (string)($cfg['admin_username'] ?? 'admin');
}

function gmap_admin_password_hash(): string
{
    $cfg = gmap_load_config();
    return (string)($cfg['admin_password_hash'] ?? '');
}

function gmap_debug_api_enabled(): bool
{
    $cfg = gmap_load_config();
    $value = $cfg['debug_api'] ?? false;
    if (is_bool($value)) {
        return $value;
    }
    $normalized = strtolower(trim((string)$value));
    return in_array($normalized, ['1', 'true', 'yes', 'on'], true);
}

function gmap_debug_key(): string
{
    $cfg = gmap_load_config();
    return trim((string)($cfg['debug_key'] ?? ''));
}

function gmap_cors_allow_origin(): string
{
    $cfg = gmap_load_config();
    $origin = trim((string)($cfg['cors_allow_origin'] ?? '*'));
    return $origin !== '' ? $origin : '*';
}

function gmap_now_utc(): string
{
    return gmdate('Y-m-d H:i:s');
}

function gmap_client_ip(): string
{
    $keys = [
        'HTTP_CF_CONNECTING_IP',
        'HTTP_X_FORWARDED_FOR',
        'HTTP_X_REAL_IP',
        'REMOTE_ADDR',
    ];
    foreach ($keys as $k) {
        if (!isset($_SERVER[$k])) {
            continue;
        }
        $raw = trim((string)$_SERVER[$k]);
        if ($raw === '') {
            continue;
        }
        if ($k === 'HTTP_X_FORWARDED_FOR') {
            $parts = explode(',', $raw);
            $raw = trim((string)($parts[0] ?? ''));
        }
        if (filter_var($raw, FILTER_VALIDATE_IP)) {
            return $raw;
        }
    }
    return '0.0.0.0';
}

function gmap_user_agent(): string
{
    $ua = trim((string)($_SERVER['HTTP_USER_AGENT'] ?? 'unknown'));
    return substr($ua, 0, 255);
}

function gmap_normalize_license_key(string $value): string
{
    $upper = strtoupper(trim($value));
    return preg_replace('/[^A-Z0-9]/', '', $upper) ?? '';
}

function gmap_hash_license_key(string $normalizedKey): string
{
    return hash('sha256', $normalizedKey);
}

function gmap_format_license_key(string $normalizedKey): string
{
    $chunks = str_split($normalizedKey, 4);
    return implode('-', $chunks);
}

function gmap_generate_license_key(): string
{
    $raw = strtoupper(bin2hex(random_bytes(8)));
    return $raw;
}

function gmap_insert_log(
    PDO $pdo,
    ?int $licenseId,
    string $eventType,
    ?string $hwidHash,
    string $message
): void {
    $stmt = $pdo->prepare(
        'INSERT INTO license_logs (license_id, event_type, time, ip, hwid_hash, user_agent, message)
         VALUES (:license_id, :event_type, :time, :ip, :hwid_hash, :user_agent, :message)'
    );
    $stmt->execute([
        ':license_id' => $licenseId,
        ':event_type' => $eventType,
        ':time' => gmap_now_utc(),
        ':ip' => gmap_client_ip(),
        ':hwid_hash' => $hwidHash,
        ':user_agent' => gmap_user_agent(),
        ':message' => substr($message, 0, 255),
    ]);
}
